<?php

namespace AgeGate\UserRegistration\Integration;

use AgeGate\Utility\Age;
use AgeGate\Utility\Cookie;
use Asylum\Validation\Validator;
use Asylum\Utility\StringTemplate;
use AgeGate\Common\Settings as CoreSettings;
use AgeGate\UserRegistration\Settings\Settings;

class Core extends AbstractIntegration
{
    public function __construct()
    {
        // Form fields
        add_action('register_form', [$this, 'fields']);

        // error messages
        add_filter('registration_errors', [$this, 'errors'], 10, 3 );

        // validation
		add_action('user_register', [$this, 'store'], 10, 1 );

        // user profile
        add_action('user_new_form', [$this, 'profile'], 10, 1);
        add_action('show_user_profile', [$this, 'profile'], 10, 1);
		add_action('edit_user_profile', [$this, 'profile'], 10, 1);
        add_action('profile_update', [$this, 'store']);

        add_action('wp_login', [$this, 'login'], 10, 2);

        add_action('wp_logout', [$this, 'logout'], 10, 1);
    }

    public function errors($errors, $login, $email)
    {
        $settings = Settings::getInstance();

        if (!$settings->restrictRegister) {
            return;
        }

        $core = CoreSettings::getInstance();

        $validation = new Validator;

        $data = $validation->sanitize($_POST);

        $minYear = apply_filters('age_gate/form/min_year', 1900);
        $maxYear = apply_filters('age_gate/form/max_year', date('Y'));

        $data['age_gate']['dob'] = (new StringTemplate())->render('{y}-{m}-{d}', $data['age_gate']);

        $rules = [
            'age_gate.d' => 'required|numeric|min_len,1|max_len,2|max_numeric,31',
            'age_gate.m' => 'required|numeric|min_len,1|max_len,2|max_numeric,12',
            'age_gate.y' => 'required|numeric|min_len,4|max_len,4|min_numeric,'. $minYear .'|max_numeric,' . $maxYear,
            'age_gate.dob' => 'required|date|min_age,' . ($settings->registerAge ?: $core->defaultAge)
        ];

        $validation->validation_rules($rules);

        $valid = $validation->run($data);

        if ($validation->errors()) {
            $errors->add('toyoungerror', '<strong>' . __('ERROR', 'age-gate-user-registration') . '</strong>: ' . sprintf($settings->registrationError, $settings->registerAge ?: $core->defaultAge));
        };


        return $errors;

    }

    public function validate()
    {}

    public function profile()
    {
        $settings = Settings::getInstance();

        if ($settings->storeDob) {
            $this->render($settings, CoreSettings::getInstance(), false);
        }
    }

    public function store($userId)
    {
        $settings = Settings::getInstance();

        if (!$settings->restrictRegister) {
            return;
        }

        $validation = new Validator;
        $data = $validation->sanitize($_POST);

        if (!isset($data['age_gate'])) {
            return;
        }

        $validated = Validator::is_valid($data['age_gate'], array(
            'd' => 'required|numeric|min_len,2|max_len,2|max_numeric,31',
            'm' => 'required|numeric|min_len,2|max_len,2|max_numeric,12',
            'y' => 'required|numeric|min_len,4|max_len,4|max_numeric,' . date('Y'),
        ));


        if ($validated === true) {
            update_user_meta( $userId, 'u_db', $data['age_gate']['y'] . '-' . $data['age_gate']['m'] . '-' . $data['age_gate']['d'] );
            update_user_meta( $userId, 'user_dob', $data['age_gate'] );
        }
    }

    public function login($login, $user)
    {
        $settings = Settings::getInstance();

        if (!$settings->storeDob || !$settings->setAge) {
            return;
        }

        $dob = get_user_meta($user->ID, 'u_db', true);

        if (!$dob) {
            return;
        }

        $core = CoreSettings::getInstance();
        Cookie::set($core->getCookieName(), Age::calculateAge($dob));
    }

    public function logout($userId)
    {
        $settings = Settings::getInstance();

        if (!$settings->setAge) {
            return;
        }

        $core = CoreSettings::getInstance();
        if (Cookie::get($core->getCookieName())) {
            Cookie::destroy($core->getCookieName());
        }

    }

}
